// Configuration
const CONFIG = {
  API_URL: 'https://palal.app/api',
  CAPTURE_INTERVAL: 10000,  // 10 seconds fixed
  AUTH_TOKEN_KEY: 'auth_token',
  USER_ID_KEY: 'user_id',
  USER_NAME_KEY: 'user_name',
  USER_EMAIL_KEY: 'user_email',
  COMPANY_ID_KEY: 'company_id',
  COMPANY_NAME_KEY: 'company_name',
  COMPANY_SLUG_KEY: 'company_slug',
  IS_TRACKING_KEY: 'is_tracking'
};

const TOKEN_REFRESH_THRESHOLD_SECONDS = 5;
let cachedTokenMetadata = null;
let accessTokenRefreshPromise = null;

function normalizeCompanyId(value) {
  if (value === undefined || value === null || value === '') {
    return null;
  }
  const numeric = Number(value);
  if (!Number.isFinite(numeric) || numeric <= 0) {
    return null;
  }
  return Math.floor(numeric);
}

function decodeJwtPayload(token) {
  if (!token) {
    return null;
  }
  const parts = token.split('.');
  if (parts.length < 2) {
    return null;
  }
  try {
    const payload = parts[1]
      .replace(/-/g, '+')
      .replace(/_/g, '/');
    const padded = payload + '='.repeat((4 - (payload.length % 4 || 4)) % 4);
    const decoded = atob(padded);
    return JSON.parse(decoded);
  } catch (error) {
    console.debug('Failed to decode JWT payload:', error.message);
    return null;
  }
}

function updateCachedTokenMetadata(token) {
  if (!token) {
    cachedTokenMetadata = null;
    return;
  }
  const payload = decodeJwtPayload(token);
  if (!payload || typeof payload.exp !== 'number') {
    cachedTokenMetadata = null;
    return;
  }
  cachedTokenMetadata = { token, exp: payload.exp };
}

function tokenExpiresSoon(token, thresholdSeconds = TOKEN_REFRESH_THRESHOLD_SECONDS) {
  if (!token) {
    return true;
  }
  if (!cachedTokenMetadata || cachedTokenMetadata.token !== token) {
    updateCachedTokenMetadata(token);
  }
  if (!cachedTokenMetadata || typeof cachedTokenMetadata.exp !== 'number') {
    return true;
  }
  const nowSeconds = Math.floor(Date.now() / 1000);
  return cachedTokenMetadata.exp - nowSeconds <= thresholdSeconds;
}

async function refreshAccessToken() {
  if (accessTokenRefreshPromise) {
    return accessTokenRefreshPromise;
  }

  accessTokenRefreshPromise = (async () => {
    const response = await fetch(`${CONFIG.API_URL}/auth/token/refresh`, {
      method: 'POST',
      credentials: 'include'
    });

    if (!response.ok) {
      throw new Error(`Refresh failed with status ${response.status}`);
    }

    const payload = await response.json();
    if (!payload.success || !payload.data?.token) {
      throw new Error(payload.message || 'Refresh response missing token');
    }

    const refreshedUser = payload.data.user || {};
    const company = refreshedUser.company || {};

    await persistAuthData({
      auth_token: payload.data.token,
      user_id: refreshedUser.id || null,
      user_name: refreshedUser.name || '',
      user_email: refreshedUser.email || '',
      company_id: refreshedUser.company_id || company.id || null,
      company_name: company.name || '',
      company_slug: company.slug || ''
    });

    return payload.data.token;
  })();

  try {
    return await accessTokenRefreshPromise;
  } finally {
    accessTokenRefreshPromise = null;
  }
}

const MANUAL_START_TOKEN_KEY = 'manual_start_token';
const MANUAL_START_REQUESTED_AT_KEY = 'manual_start_requested_at';
const MANUAL_START_TTL_MS = 15000;

let captureTimeout = null;
let isTracking = false;
let hasScreenPermission = false;
let lastStreamId = null; // Store the stream ID
let sessionStartTime = null;
let totalActiveTime = 0; // Total milliseconds active in current session
let sessionTrackingInterval = null;
let startTrackingInProgress = false;
let lastCaptureTime = 0;
let pendingMouseClicks = 0;
let lastMouseActivityTs = 0;
const DEFAULT_ACTION_ICON_PATHS = Object.freeze({
  16: 'icons/icon16.png',
  48: 'icons/icon48.png',
  128: 'icons/icon128.png'
});
let cachedActiveIconData = null;
let activeIconGenerationPromise = null;

async function ensureClickTrackerRegistered() {
  if (!chrome.scripting || !chrome.scripting.registerContentScripts) {
    console.warn('chrome.scripting API unavailable, mouse clicks will not be tracked.');
    return;
  }

  try {
    const existing = await chrome.scripting.getRegisteredContentScripts({ ids: ['click-tracker'] });
    if (existing && existing.length > 0) {
      return;
    }
  } catch (error) {
    if (error?.message && !error.message.includes('No content script with id')) {
      console.warn('Unable to verify click tracker registration:', error.message);
    }
  }

  try {
    await chrome.scripting.registerContentScripts([
      {
        id: 'click-tracker',
        js: ['clickTracker.js'],
        matches: ['<all_urls>'],
        allFrames: true,
        runAt: 'document_start',
        persistAcrossSessions: true
      }
    ]);
    console.log('Click tracker script registered.');
  } catch (error) {
    console.error('Failed to register click tracker content script:', error);
  }
}

function consumeMouseClicks() {
  const clicks = pendingMouseClicks;
  pendingMouseClicks = 0;
  return clicks;
}

async function persistAuthData(authData, { resetTrackingFlag = false } = {}) {
  if (!authData || !authData.auth_token) {
    return false;
  }

  const companyId = normalizeCompanyId(authData.company_id);
  const payload = {
    [CONFIG.AUTH_TOKEN_KEY]: authData.auth_token,
    [CONFIG.USER_ID_KEY]: authData.user_id || null,
    [CONFIG.USER_NAME_KEY]: authData.user_name || '',
    [CONFIG.USER_EMAIL_KEY]: authData.user_email || '',
    [CONFIG.COMPANY_ID_KEY]: companyId,
    [CONFIG.COMPANY_NAME_KEY]: authData.company_name || '',
    [CONFIG.COMPANY_SLUG_KEY]: authData.company_slug || ''
  };

  if (resetTrackingFlag) {
    payload[CONFIG.IS_TRACKING_KEY] = false;
    isTracking = false;
  }

  updateCachedTokenMetadata(authData.auth_token);
  await chrome.storage.local.set(payload);

  if (resetTrackingFlag) {
    await updateActionIcon(false);
  }

  return true;
}

async function syncAuthFromTab(tabId) {
  try {
    const authData = await chrome.tabs.sendMessage(tabId, { action: 'get_local_storage' });
    if (authData && authData.auth_token) {
      const stored = await persistAuthData(authData);
      if (stored) {
        console.log('Synced auth data from tab', authData.user_email || authData.user_name || 'unknown user');
        try {
          await chrome.runtime.sendMessage({ action: 'login_success' });
        } catch (e) {
          // Popup may not be open; ignore errors
        }
      }
      return stored;
    }
  } catch (error) {
    if (error?.message && error.message.includes('Receiving end does not exist')) {
      // Content script not ready on this tab yet; ignore quietly
      return false;
    }
    console.debug('Auth sync skipped for tab', tabId, '-', error.message);
  }
  return false;
}

async function generateIconImageData(path, size, withIndicator = false) {
  try {
    const response = await fetch(chrome.runtime.getURL(path));
    if (!response.ok) {
      throw new Error(`Failed to load icon: ${path}`);
    }
    const blob = await response.blob();
    const bitmap = await createImageBitmap(blob);
    const canvas = new OffscreenCanvas(size, size);
    const ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, size, size);
    ctx.drawImage(bitmap, 0, 0, size, size);

    if (withIndicator) {
      const baseRadius = Math.max(2, Math.round(size / 6));
      const radius = Math.min(Math.round(size / 2) - 1, baseRadius * 2);
      const margin = Math.max(1, Math.round(size / 20));
      const centerX = size - radius - margin;
      const centerY = size - radius - margin;
      ctx.beginPath();
      ctx.fillStyle = '#22c55e';
      ctx.strokeStyle = '#ffffff';
      ctx.lineWidth = Math.max(1, Math.round(size / 18));
      ctx.arc(centerX, centerY, radius, 0, Math.PI * 2);
      ctx.fill();
      ctx.stroke();
    }

    return ctx.getImageData(0, 0, size, size);
  } catch (error) {
    console.warn('Icon generation failed:', error.message);
    return null;
  }
}

async function getActiveIconImageData() {
  if (cachedActiveIconData) {
    return cachedActiveIconData;
  }
  if (!activeIconGenerationPromise) {
    activeIconGenerationPromise = (async () => {
      const imageDataMap = {};
      for (const [sizeKey, path] of Object.entries(DEFAULT_ACTION_ICON_PATHS)) {
        const size = Number(sizeKey);
        const imageData = await generateIconImageData(path, size, true);
        if (imageData) {
          imageDataMap[size] = imageData;
        }
      }
      if (Object.keys(imageDataMap).length === 0) {
        throw new Error('No icon image data generated');
      }
      cachedActiveIconData = imageDataMap;
      return cachedActiveIconData;
    })().catch((error) => {
      console.error('Failed to prepare active icon image data:', error);
      activeIconGenerationPromise = null;
      throw error;
    });
  }
  return activeIconGenerationPromise;
}

async function updateActionIcon(active) {
  try {
    if (active) {
      const imageData = await getActiveIconImageData();
      await chrome.action.setIcon({ imageData });
    } else {
      await chrome.action.setIcon({ path: DEFAULT_ACTION_ICON_PATHS });
    }
  } catch (error) {
    console.warn('Unable to update action icon:', error.message);
  }
}
async function isManualStartTokenValid(token) {
  if (!token) {
    console.warn('Missing manual start token. Rejecting start request.');
    return false;
  }

  const state = await chrome.storage.local.get([
    MANUAL_START_TOKEN_KEY,
    MANUAL_START_REQUESTED_AT_KEY
  ]);

  if (!state[MANUAL_START_TOKEN_KEY] || state[MANUAL_START_TOKEN_KEY] !== token) {
    console.warn('Manual start token mismatch.');
    return false;
  }

  const requestedAt = state[MANUAL_START_REQUESTED_AT_KEY] || 0;
  const age = Date.now() - requestedAt;
  if (age > MANUAL_START_TTL_MS) {
    console.warn(`Manual start token expired after ${age}ms.`);
    return false;
  }

  return true;
}

async function clearManualStartToken() {
  await chrome.storage.local.remove([
    MANUAL_START_TOKEN_KEY,
    MANUAL_START_REQUESTED_AT_KEY
  ]);
}

// Get employee info from storage
async function getEmployeeInfo(options = {}) {
  const { ensureFreshToken = true } = options;
  const keys = [
    CONFIG.AUTH_TOKEN_KEY,
    CONFIG.USER_ID_KEY,
    CONFIG.USER_NAME_KEY,
    CONFIG.USER_EMAIL_KEY,
    CONFIG.COMPANY_ID_KEY,
    CONFIG.COMPANY_NAME_KEY,
    CONFIG.COMPANY_SLUG_KEY
  ];

  let state = await chrome.storage.local.get(keys);
  let authToken = state[CONFIG.AUTH_TOKEN_KEY] || null;

  if (authToken) {
    updateCachedTokenMetadata(authToken);
    if (ensureFreshToken && tokenExpiresSoon(authToken)) {
      try {
        await refreshAccessToken();
        state = await chrome.storage.local.get(keys);
        authToken = state[CONFIG.AUTH_TOKEN_KEY] || null;
        if (authToken) {
          updateCachedTokenMetadata(authToken);
        }
      } catch (error) {
        console.error('Unable to refresh auth token:', error.message);
        authToken = null;
      }
    }
  } else {
    updateCachedTokenMetadata(null);
  }

  const companyId = normalizeCompanyId(state[CONFIG.COMPANY_ID_KEY]);
  return {
    auth_token: authToken,
    user_id: state[CONFIG.USER_ID_KEY] || null,
    name: state[CONFIG.USER_NAME_KEY] || '',
    email: state[CONFIG.USER_EMAIL_KEY] || '',
    company_id: companyId,
    company_name: state[CONFIG.COMPANY_NAME_KEY] || '',
    company_slug: state[CONFIG.COMPANY_SLUG_KEY] || ''
  };
}

// Request screen capture permission
async function requestScreenPermission() {
  try {
    console.log('🔵 Starting screen permission request...');
    
    // Ensure offscreen document exists FIRST
    await createOffscreenDocument();
    await new Promise(resolve => setTimeout(resolve, 100));
    console.log('✅ Offscreen document ready');
    
    // Use getDisplayMedia instead of desktopCapture for Manifest V3
    // Send message to offscreen to request permission there
    console.log('📺 Requesting display media permission from offscreen...');
    const response = await chrome.runtime.sendMessage({
      type: 'request_display_media'
    });
    
    console.log('📺 Display media response:', response);
    
    if (response && response.success) {
      hasScreenPermission = true;
      lastStreamId = 'display-media-active'; // Marker that stream is active
      await chrome.storage.local.set({ screen_permission_granted: true });
      console.log('✅ Screen capture stream started successfully');
      return lastStreamId;
    } else {
      hasScreenPermission = false;
      lastStreamId = null;
      throw new Error(response?.error || 'Failed to start screen capture');
    }
    
  } catch (error) {
    console.error('❌ Screen permission error:', error);
    hasScreenPermission = false;
    lastStreamId = null;
    throw error;
  }
}

// Track if offscreen document is already created
let offscreenDocumentCreated = false;

// Create offscreen document for screen capture
async function createOffscreenDocument() {
  // Check if already created in memory
  if (offscreenDocumentCreated) {
    return;
  }
  
  try {
    // Check if offscreen document already exists
    const existingContexts = await chrome.runtime.getContexts({
      contextTypes: ['OFFSCREEN_DOCUMENT']
    });

    if (existingContexts.length > 0) {
      offscreenDocumentCreated = true;
      return;
    }

    // Create new offscreen document
    await chrome.offscreen.createDocument({
      url: 'offscreen.html',
      reasons: ['USER_MEDIA'],
      justification: 'Capture screen for employee tracking'
    });
    
    offscreenDocumentCreated = true;
    console.log('Offscreen document created successfully');
  } catch (error) {
    // If error is about document already existing, mark as created
    if (error.message.includes('Only a single offscreen document')) {
      offscreenDocumentCreated = true;
      console.log('Offscreen document already exists');
    } else {
      console.error('Error creating offscreen document:', error);
    }
  }
}

// Capture screenshot of the entire screen
async function captureScreenshot() {
  try {
    console.log('captureScreenshot called');
    
    // Check if screen capture permission is granted and we have a valid stream ID
    const permissionResult = await chrome.storage.local.get(['screen_permission_granted']);
    
    if (permissionResult.screen_permission_granted && lastStreamId) {
      console.log('Using desktop capture (full screen) with saved permission');
      try {
        const screenshot = await captureScreenFromStream();
        console.log('Desktop screenshot captured, length:', screenshot.length);
        return screenshot;
      } catch (error) {
        console.error('Desktop capture failed:', error.message);
        console.log('Stream might be invalid, clearing permission');
        // Clear the invalid permissions
        lastStreamId = null;
        hasScreenPermission = false;
        await chrome.storage.local.set({ screen_permission_granted: false });
        console.log('Falling back to tab capture');
      }
    }
    
    // Fallback to tab capture
    console.log('Using tab capture (browser window only)');
    
    // Get active tab first
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tabs || tabs.length === 0) {
      console.error('No active tab found');
      return createFallbackScreenshot('No active tab');
    }
    
    const activeTab = tabs[0];
    console.log('Active tab:', activeTab.id, activeTab.url);
    
    // Try to capture active tab
    try {
      const screenshot = await chrome.tabs.captureVisibleTab(activeTab.windowId, {
        format: 'png',
        quality: 80
      });
      
      console.log('Tab screenshot captured, length:', screenshot.length);
      return screenshot;
    } catch (tabError) {
      console.error('Tab capture failed:', tabError.message);
      return createFallbackScreenshot(tabError.message);
    }
  } catch (error) {
    console.error('Error capturing screenshot:', error);
    return createFallbackScreenshot(error.message);
  }
}

// Create fallback test screenshot
async function createFallbackScreenshot(reason) {
  console.log('Creating fallback screenshot. Reason:', reason);
  const canvas = new OffscreenCanvas(800, 600);
  const ctx = canvas.getContext('2d');
  
  // Draw test pattern
  ctx.fillStyle = '#f0f0f0';
  ctx.fillRect(0, 0, 800, 600);
  ctx.fillStyle = '#333';
  ctx.font = '24px Arial';
  ctx.fillText('Screenshot at ' + new Date().toLocaleTimeString(), 50, 280);
  ctx.font = '14px Arial';
  ctx.fillText('Reason: ' + reason, 50, 320);
  
  const blob = await canvas.convertToBlob({ type: 'image/png', quality: 0.8 });
  const reader = new FileReader();
  
  return new Promise((resolve) => {
    reader.onloadend = () => resolve(reader.result);
    reader.readAsDataURL(blob);
  });
}

// Capture screen using offscreen document and MediaStream
async function captureScreenFromStream() {
  return new Promise(async (resolve, reject) => {
    try {
      // Use the stored stream ID (already granted by user)
      if (!lastStreamId) {
        reject(new Error('No screen permission granted'));
        return;
      }

      // Ensure offscreen document exists
      await createOffscreenDocument();

      // Capture a frame from the active stream
      const response = await chrome.runtime.sendMessage({
        type: 'capture_frame'
      });

      if (response && response.screenshot) {
        resolve(response.screenshot);
      } else {
        reject(new Error(response?.error || 'Failed to capture frame from offscreen'));
      }
    } catch (error) {
      reject(error);
    }
  });
}

// Get browser info
function getBrowserInfo() {
  return navigator.userAgent;
}

// Get screen resolution (works in service worker)
function getScreenResolution() {
  // Service workers don't have access to screen object
  // We'll get it from the active tab or use a default
  return '1920x1080'; // Default, will be updated when capturing from tab
}

// Upload screenshot to API
async function uploadScreenshot(screenshotData, mouseClicks = null) {
  try {
    console.log('=== uploadScreenshot called ===');
    console.log('Screenshot data length:', screenshotData?.length);
    
    const employeeInfo = await getEmployeeInfo({ ensureFreshToken: true });
    console.log('Auth token present:', !!employeeInfo.auth_token);
    console.log('User ID:', employeeInfo.user_id);
    console.log('Company ID:', employeeInfo.company_id);
    
    // Check if user is authenticated
    if (!employeeInfo.auth_token || !employeeInfo.user_id) {
      console.error('User not authenticated. Cannot upload screenshot.');
      return { success: false, error: 'Not authenticated' };
    }
    if (!employeeInfo.company_id) {
      console.error('Missing company context. Cannot upload screenshot.');
      return { success: false, error: 'Workspace not selected' };
    }
    
    const payload = {
      screenshot_data: screenshotData,
      browser_info: getBrowserInfo(),
      screen_resolution: getScreenResolution()
    };

    payload.company_id = employeeInfo.company_id;
    if (employeeInfo.company_slug) {
      payload.company_slug = employeeInfo.company_slug;
    }
    if (employeeInfo.company_name) {
      payload.company_name = employeeInfo.company_name;
    }

    if (typeof mouseClicks === 'number' && mouseClicks >= 0) {
      payload.mouse_clicks = mouseClicks;
    }

    console.log('Uploading to:', `${CONFIG.API_URL}/upload`);
    console.log('Payload size:', JSON.stringify(payload).length, 'bytes');
    console.log('Browser info:', payload.browser_info);
    console.log('Screen resolution:', payload.screen_resolution);
    if (payload.mouse_clicks !== undefined) {
      console.log('Mouse clicks for interval:', payload.mouse_clicks);
    }

    const startTime = Date.now();
    const response = await fetch(`${CONFIG.API_URL}/upload`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${employeeInfo.auth_token}`
      },
      body: JSON.stringify(payload)
    });

    const uploadDuration = Date.now() - startTime;
    console.log('Upload took:', uploadDuration, 'ms');
    console.log('Response status:', response.status);
    console.log('Response headers:', Object.fromEntries(response.headers.entries()));
    
    const responseText = await response.text();
    console.log('Raw response:', responseText);
    
    let result;
    try {
      result = JSON.parse(responseText);
      console.log('Parsed response:', JSON.stringify(result, null, 2));
    } catch (e) {
      console.error('Failed to parse response as JSON:', e);
      return { success: false, error: 'Invalid JSON response', raw: responseText };
    }
    
    if (response.status === 401) {
      // Token expired or invalid
      console.error('❌ Authentication failed. Token expired or invalid.');
      // Clear auth data
      await chrome.storage.local.remove([
        CONFIG.AUTH_TOKEN_KEY,
        CONFIG.USER_ID_KEY,
        CONFIG.USER_NAME_KEY,
        CONFIG.USER_EMAIL_KEY,
        CONFIG.COMPANY_ID_KEY,
        CONFIG.COMPANY_NAME_KEY,
        CONFIG.COMPANY_SLUG_KEY
      ]);
      updateCachedTokenMetadata(null);
      await stopTracking();
      return { success: false, error: 'Authentication expired' };
    }
    
    if (result.success) {
      console.log('✅ Screenshot uploaded successfully!');
      console.log('Screenshot ID:', result.screenshot_id || result.data?.id);
      console.log('API Message:', result.message);
      if (result.data) {
        console.log('Response data:', JSON.stringify(result.data, null, 2));
      }
      updateLastUploadTime();
    } else {
      console.error('❌ Failed to upload screenshot');
      console.error('Error message:', result.message);
      console.error('Error details:', result.error || result.errors || 'No details');
    }

    return result;
  } catch (error) {
    console.error('❌ Exception during upload:', error);
    console.error('Error stack:', error.stack);
    return { success: false, error: error.message };
  }
}

// Update last upload timestamp
function updateLastUploadTime() {
  chrome.storage.local.get(['upload_count']).then(result => {
    chrome.storage.local.set({ 
      last_upload: new Date().toISOString(),
      upload_count: (result.upload_count || 0) + 1
    });
  });
}

// Get upload count
async function getUploadCount() {
  const result = await chrome.storage.local.get(['upload_count']);
  return result.upload_count || 0;
}

// Main capture and upload function
async function captureAndUpload(trigger = 'schedule') {
  const logMsg = (msg) => {
    console.log(msg);
    // Send log to popup if open
    chrome.runtime.sendMessage({ action: 'log', message: msg }).catch(() => {});
  };
  
  logMsg('=== Starting captureAndUpload ===');
  logMsg('isTracking: ' + isTracking);
  
  if (!isTracking) {
    logMsg('Tracking disabled, skipping capture request.');
    return false;
  }

  const now = Date.now();
  const elapsed = now - (lastCaptureTime || 0);
  const minGap = CONFIG.CAPTURE_INTERVAL - 500; // small buffer so we never exceed server expectations

  if (lastCaptureTime && elapsed < minGap) {
    const wait = Math.max(CONFIG.CAPTURE_INTERVAL - elapsed, 1000);
    logMsg(`⏱️ Capture throttled (${trigger}). Only ${elapsed}ms since last capture. Scheduling in ${wait}ms.`);
    scheduleNextCapture(wait);
    return false;
  }

  lastCaptureTime = now;

  try {
    const screenshot = await captureScreenshot();
    logMsg('📸 Screenshot captured, size: ' + (screenshot ? (screenshot.length / 1024).toFixed(2) + ' KB' : 'null'));
    
    if (screenshot) {
      const mouseClicksForInterval = consumeMouseClicks();
      logMsg('🖱️ Mouse clicks captured this interval: ' + mouseClicksForInterval);
      logMsg('📤 Uploading to API...');
      const result = await uploadScreenshot(screenshot, mouseClicksForInterval);
      
      if (result.success) {
        logMsg('✅ Upload SUCCESS - ID: ' + (result.data?.screenshot_id || 'unknown'));
        if (result.data?.url) {
          logMsg('🔗 Screenshot URL: ' + result.data.url);
        }
      } else {
        logMsg('❌ Upload FAILED - ' + (result.message || result.error || 'Unknown error'));
        if ((result.error || '').toLowerCase() === 'not authenticated') {
          logMsg('⚠️ Authentication missing. Stopping tracking loop.');
          await stopTracking();
          return false;
        }
      }
    } else {
      logMsg('❌ Screenshot capture returned null');
    }
  } catch (error) {
    logMsg('💥 Exception: ' + error.message);
    console.error('Error in captureAndUpload:', error);
  }
  
  // Schedule next capture if tracking is enabled
  if (isTracking) {
    scheduleNextCapture();
  }

  return true;
}

// Schedule next capture with fixed interval
function scheduleNextCapture(customDelay = null) {
  const interval = Math.max(customDelay ?? CONFIG.CAPTURE_INTERVAL, 1000);
  console.log(`Next capture in ${(interval / 1000).toFixed(1)} seconds`);
  
  // Clear existing timeout
  if (captureTimeout) {
    clearTimeout(captureTimeout);
  }
  
  // Schedule next capture
  captureTimeout = setTimeout(captureAndUpload, interval);
  
  // Save next capture time
  chrome.storage.local.set({
    next_capture: new Date(Date.now() + interval).toISOString()
  });
}

// Start tracking
async function startTracking() {
  console.log('=== startTracking called ===');
  if (startTrackingInProgress) {
    console.log('Start request already in progress. Skipping.');
    return false;
  }

  if (isTracking) {
    console.log('Tracking is already active.');
    return false;
  }
  
  // Check if user is authenticated
  const employeeInfo = await getEmployeeInfo({ ensureFreshToken: true });
  console.log('Employee info:', {
    hasToken: !!employeeInfo.auth_token,
    hasUserId: !!employeeInfo.user_id,
    userId: employeeInfo.user_id
  });
  
  if (!employeeInfo.auth_token || !employeeInfo.user_id) {
    console.log('User not authenticated. Cannot start tracking.');
    isTracking = false;
    await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
    await updateActionIcon(false);
    return false;
  }
  if (!employeeInfo.company_id) {
    console.log('Company context missing. Prompt user to log in again.');
    isTracking = false;
    await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
    await updateActionIcon(false);
    return false;
  }
  
  startTrackingInProgress = true;
  
  try {
    // DON'T auto-request screen permission - let user click the button
    // Screen permission must be granted by user interaction with popup
    let canProceed = true;
    
    // Call API to start session - this stores session_start_at in database
    try {
      const response = await fetch(`${CONFIG.API_URL}/session/start`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${employeeInfo.auth_token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          company_id: employeeInfo.company_id,
          company_slug: employeeInfo.company_slug || undefined
        })
      });
      
      const result = await response.json();
      console.log('📍 Session start API response:', result);
      const alreadyActive = (result.message || '').toLowerCase().includes('active session');
      
      if (!result.success && !alreadyActive) {
        console.error('Failed to start session:', result.message);
        canProceed = false;
      } else if (!result.success && alreadyActive) {
        console.log('⚠️ Active session exists, resuming local tracking loop');
      } else {
        console.log('✅ Session started on server:', result.data);
      }
    } catch (error) {
      console.error('Error starting session:', error);
      // Continue with local tracking even if API fails
    }
    
    if (!canProceed) {
      return false;
    }
    
    // Start local session timer
    sessionStartTime = Date.now();
    console.log('Session started at:', new Date(sessionStartTime).toLocaleString());
    
    // Force enable tracking
    isTracking = true;
    await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: true });
    await updateActionIcon(true);
    
    // Start tracking active time
    startSessionTracking();
    
    console.log('Starting screenshot tracking...');
    console.log('Will capture immediately, then every 10 seconds');
    
    // Capture immediately
    await captureAndUpload('initial');
    return true;
  } finally {
    startTrackingInProgress = false;
  }
}

// Stop tracking
async function stopTracking() {
  console.log('Stopping screenshot tracking...');
  
  // Stop screen capture stream if active
  if (hasScreenPermission && lastStreamId) {
    try {
      await chrome.runtime.sendMessage({ type: 'stop_screen_capture' });
      console.log('Screen capture stream stopped');
    } catch (error) {
      console.error('Error stopping screen capture:', error);
    }
    lastStreamId = null;
    hasScreenPermission = false;
    await chrome.storage.local.set({ screen_permission_granted: false });
  }
  
  // Get employee info for API call
  const employeeInfo = await getEmployeeInfo({ ensureFreshToken: false });
  
  // Call API to end session - this stores session_end_at and calculates duration
  if (employeeInfo.auth_token) {
    try {
      const response = await fetch(`${CONFIG.API_URL}/session/end`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${employeeInfo.auth_token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          company_id: employeeInfo.company_id || undefined,
          company_slug: employeeInfo.company_slug || undefined
        })
      });
      
      const result = await response.json();
      console.log('🛑 Session end API response:', result);
      
      if (result.success) {
        console.log('✅ Session ended on server:', result.data);
        console.log('📊 Total duration:', result.data.duration_ms, 'ms (', Math.floor(result.data.duration_ms / 60000), 'minutes )');
        console.log('📸 Screenshots captured:', result.data.screenshot_count);
      } else {
        console.error('Failed to end session:', result.message);
      }
    } catch (error) {
      console.error('Error ending session:', error);
    }
  }
  
  // Calculate local session duration
  if (sessionStartTime) {
    const sessionDuration = Date.now() - sessionStartTime;
    totalActiveTime += sessionDuration;
    console.log('Local session duration:', Math.floor(sessionDuration / 1000), 'seconds');
    sessionStartTime = null;
  }
  
  isTracking = false;
  if (sessionTrackingInterval) {
    clearInterval(sessionTrackingInterval);
    sessionTrackingInterval = null;
  }
  
  if (captureTimeout) {
    clearTimeout(captureTimeout);
    captureTimeout = null;
  }
  lastCaptureTime = 0;
  
  await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
  await updateActionIcon(false);
  return true;
}

// Session time tracking
function startSessionTracking() {
  if (sessionTrackingInterval) {
    clearInterval(sessionTrackingInterval);
  }
  // Update active time every minute
  sessionTrackingInterval = setInterval(() => {
    if (isTracking && sessionStartTime) {
      const currentDuration = Date.now() - sessionStartTime;
      console.log('Current session:', Math.floor(currentDuration / 1000), 'seconds');
    }
  }, 60000); // Log every minute
}

// Save session time to storage
async function saveSessionTime(duration) {
  const today = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
  const thisWeek = getWeekKey();
  const thisMonth = new Date().toISOString().substring(0, 7); // YYYY-MM
  
  const result = await chrome.storage.local.get(['session_times']);
  const sessionTimes = result.session_times || {};
  
  // Initialize structures if needed
  if (!sessionTimes.daily) sessionTimes.daily = {};
  if (!sessionTimes.weekly) sessionTimes.weekly = {};
  if (!sessionTimes.monthly) sessionTimes.monthly = {};
  
  // Add to daily total
  sessionTimes.daily[today] = (sessionTimes.daily[today] || 0) + duration;
  
  // Add to weekly total
  sessionTimes.weekly[thisWeek] = (sessionTimes.weekly[thisWeek] || 0) + duration;
  
  // Add to monthly total
  sessionTimes.monthly[thisMonth] = (sessionTimes.monthly[thisMonth] || 0) + duration;
  
  // Save back to storage
  await chrome.storage.local.set({ session_times: sessionTimes });
  
  console.log('Session time saved locally:', {
    duration: Math.floor(duration / 1000) + 's',
    today: Math.floor(sessionTimes.daily[today] / 1000) + 's',
    thisWeek: Math.floor(sessionTimes.weekly[thisWeek] / 1000) + 's',
    thisMonth: Math.floor(sessionTimes.monthly[thisMonth] / 1000) + 's'
  });
  
  // Sync to server
  await syncSessionTimeToServer(today, sessionTimes.daily[today]);
}

// Get week key (year-week format)
function getWeekKey() {
  const now = new Date();
  const start = new Date(now.getFullYear(), 0, 1);
  const diff = now - start;
  const oneWeek = 1000 * 60 * 60 * 24 * 7;
  const week = Math.floor(diff / oneWeek);
  return `${now.getFullYear()}-W${week.toString().padStart(2, '0')}`;
}

// Sync session time to server
async function syncSessionTimeToServer(date, durationMs) {
  try {
    const employeeInfo = await getEmployeeInfo({ ensureFreshToken: true });
    
    if (!employeeInfo.auth_token || !employeeInfo.user_id) {
      console.log('Not syncing session time - user not authenticated');
      return;
    }
    if (!employeeInfo.company_id) {
      console.log('Not syncing session time - missing company context');
      return;
    }
    
    console.log('Syncing session time to server:', {
      date: date,
      duration: Math.floor(durationMs / 1000) + 's'
    });
    
    const response = await fetch(`${CONFIG.API_URL}/session-time`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${employeeInfo.auth_token}`
      },
      body: JSON.stringify({
        session_date: date,
        duration_ms: Math.floor(durationMs),
        company_id: employeeInfo.company_id,
        company_slug: employeeInfo.company_slug || undefined
      })
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('✅ Session time synced to server');
    } else {
      console.error('❌ Failed to sync session time:', result.message);
    }
  } catch (error) {
    console.error('Error syncing session time:', error);
  }
}

// Get session statistics
async function getSessionStats() {
  const today = new Date().toISOString().split('T')[0];
  const thisWeek = getWeekKey();
  const thisMonth = new Date().toISOString().substring(0, 7);
  
  const result = await chrome.storage.local.get(['session_times']);
  const sessionTimes = result.session_times || { daily: {}, weekly: {}, monthly: {} };
  
  return {
    today: sessionTimes.daily?.[today] || 0,
    thisWeek: sessionTimes.weekly?.[thisWeek] || 0,
    thisMonth: sessionTimes.monthly?.[thisMonth] || 0,
    currentSession: sessionStartTime ? Date.now() - sessionStartTime : 0
  };
}

// Listen for messages from popup and offscreen
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'start_tracking') {
    (async () => {
      const tokenValid = await isManualStartTokenValid(request.manualStartToken);
      if (!tokenValid) {
        sendResponse({ success: false, message: 'Start request expired. Open the popup and try again.' });
        return;
      }

      const started = await startTracking();
      await clearManualStartToken();

      if (started) {
        sendResponse({ success: true, message: 'Tracking started' });
      } else {
        sendResponse({ success: false, message: 'Unable to start tracking. Check authentication or existing session state.' });
      }
    })().catch((error) => {
      console.error('Failed to start tracking:', error);
      sendResponse({ success: false, message: error.message || 'Failed to start tracking' });
    });
    return true;
  } else if (request.action === 'stop_tracking') {
    stopTracking().then(() => {
      sendResponse({ success: true, message: 'Tracking stopped' });
    }).catch((error) => {
      console.error('Failed to stop tracking:', error);
      sendResponse({ success: false, message: error.message || 'Failed to stop tracking' });
    });
    return true;
  } else if (request.action === 'get_status') {
    chrome.storage.local.get([
      CONFIG.IS_TRACKING_KEY,
      'last_upload',
      'next_capture',
      'upload_count',
      CONFIG.USER_ID_KEY,
      'screen_permission_granted'
    ]).then(async result => {
      const sessionStats = await getSessionStats();
      sendResponse({
        isTracking: isTracking,
        lastUpload: result.last_upload,
        nextCapture: result.next_capture,
        uploadCount: result.upload_count || 0,
        userId: result[CONFIG.USER_ID_KEY],
        hasPermission: result.screen_permission_granted || false,
        sessionStats: sessionStats,
        sessionStartTime: sessionStartTime // Add session start time for timer
      });
    });
    return true; // Keep channel open for async response
  } else if (request.action === 'update_employee_info') {
    // This is deprecated - info comes from login now
    sendResponse({ success: true });
  } else if (request.action === 'capture_now') {
    captureAndUpload('manual').then((didCapture) => {
      if (didCapture) {
        sendResponse({ success: true, message: 'Capturing screenshot...' });
      } else {
        sendResponse({ success: true, message: 'Capture scheduled after throttle' });
      }
    }).catch((error) => {
      console.error('Manual capture failed:', error);
      sendResponse({ success: false, message: error.message || 'Capture failed' });
    });
    return true;
  } else if (request.action === 'click_tracker_increment') {
    const increment = Number(request.count) || 0;
    if (increment > 0) {
      pendingMouseClicks += increment;
      lastMouseActivityTs = Date.now();
    }
    sendResponse({ success: true, pendingMouseClicks });
    return false;
  } else if (request.action === 'request_permission') {
    // Request screen capture permission
    requestScreenPermission()
      .then(() => {
        sendResponse({ success: true, message: 'Screen permission granted' });
      })
      .catch((error) => {
        sendResponse({ success: false, error: error.message });
      });
    return true; // Keep channel open for async response
  } else if (request.action === 'set_auth') {
    // Store authentication data from login page
    console.log('Received set_auth request:', request);
    persistAuthData({
      auth_token: request.auth_token,
      user_id: request.user_id,
      user_name: request.user_name,
      user_email: request.user_email,
      company_id: request.company_id,
      company_name: request.company_name,
      company_slug: request.company_slug
    }, { resetTrackingFlag: true }).then((stored) => {
      if (stored) {
        console.log('Auth data stored successfully. Awaiting user action to start tracking.');
        sendResponse({ success: true, message: 'Authentication stored' });
      } else {
        sendResponse({ success: false, error: 'Missing auth token in response' });
      }
    }).catch((error) => {
      console.error('Error storing auth data:', error);
      sendResponse({ success: false, error: error.message });
    });

    return true; // Keep channel open for async response
  }
  
  return false;
});

// Initialize on installation
chrome.runtime.onInstalled.addListener(async () => {
  console.log('Employee Tracker extension installed');
  await createOffscreenDocument();
  await ensureClickTrackerRegistered();
  // Don't auto-start tracking - user needs to login first
  await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
  await updateActionIcon(false);
});

// Start tracking when extension loads
chrome.runtime.onStartup.addListener(async () => {
  console.log('Browser started, checking for active session...');
  await createOffscreenDocument();
  await ensureClickTrackerRegistered();
  const storedState = await chrome.storage.local.get([
    CONFIG.AUTH_TOKEN_KEY,
    CONFIG.IS_TRACKING_KEY
  ]);
  if (!storedState[CONFIG.AUTH_TOKEN_KEY]) {
    console.log('No auth token cached, skip auto-resume.');
    await updateActionIcon(false);
    return;
  }

  if (!storedState[CONFIG.IS_TRACKING_KEY]) {
    console.log('Tracking flag is off, waiting for user to start manually.');
    await updateActionIcon(false);
    return;
  }
  
  console.log('Auto-resume disabled. Clearing stale tracking flag.');
  await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
  await updateActionIcon(false);
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  const url = changeInfo.url || tab?.url || '';
  if (!url.startsWith('https://palal.app/')) {
    return;
  }
  if (changeInfo.status && changeInfo.status !== 'complete') {
    return;
  }
  syncAuthFromTab(tabId).catch((error) => {
    console.debug('Auth sync error for tab', tabId, '-', error?.message || error);
  });
});

// Initialize immediately (only once)
let initialized = false;
(async () => {
  if (initialized) return;
  initialized = true;
  
  console.log('Initializing extension...');
  await createOffscreenDocument();
  await ensureClickTrackerRegistered();
  await updateActionIcon(false);
  
  // Check if user is logged in and tracking was previously enabled
  const result = await chrome.storage.local.get([
    CONFIG.AUTH_TOKEN_KEY,
    CONFIG.IS_TRACKING_KEY
  ]);
  
  if (result[CONFIG.AUTH_TOKEN_KEY]) {
    updateCachedTokenMetadata(result[CONFIG.AUTH_TOKEN_KEY]);
    console.log('User authenticated');

    if (result[CONFIG.IS_TRACKING_KEY]) {
      console.log('Auto-start is disabled; clearing stale tracking flag.');
      await chrome.storage.local.set({ [CONFIG.IS_TRACKING_KEY]: false });
      await updateActionIcon(false);
    } else {
      console.log('Tracking currently stopped; awaiting manual start.');
    }
  } else {
    updateCachedTokenMetadata(null);
    console.log('User not authenticated, waiting for login...');
  }
})();
